import { ErrorCode, isDirectusError } from '@directus/errors';
import express from 'express';
import { respond } from '../middleware/respond.js';
import useCollection from '../middleware/use-collection.js';
import { validateBatch } from '../middleware/validate-batch.js';
import { DashboardsService } from '../services/dashboards.js';
import { MetaService } from '../services/meta.js';
import asyncHandler from '../utils/async-handler.js';
import { sanitizeQuery } from '../utils/sanitize-query.js';
const router = express.Router();
router.use(useCollection('directus_dashboards'));
router.post('/', asyncHandler(async (req, res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const savedKeys = [];
    if (Array.isArray(req.body)) {
        const keys = await service.createMany(req.body);
        savedKeys.push(...keys);
    }
    else {
        const key = await service.createOne(req.body);
        savedKeys.push(key);
    }
    try {
        if (Array.isArray(req.body)) {
            const items = await service.readMany(savedKeys, req.sanitizedQuery);
            res.locals['payload'] = { data: items };
        }
        else {
            const item = await service.readOne(savedKeys[0], req.sanitizedQuery);
            res.locals['payload'] = { data: item };
        }
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
const readHandler = asyncHandler(async (req, res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const metaService = new MetaService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const records = await service.readByQuery(req.sanitizedQuery);
    const meta = await metaService.getMetaForQuery(req.collection, req.sanitizedQuery);
    res.locals['payload'] = { data: records || null, meta };
    return next();
});
router.get('/', validateBatch('read'), readHandler, respond);
router.search('/', validateBatch('read'), readHandler, respond);
router.get('/:pk', asyncHandler(async (req, res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const record = await service.readOne(req.params['pk'], req.sanitizedQuery);
    res.locals['payload'] = { data: record || null };
    return next();
}), respond);
router.patch('/', validateBatch('update'), asyncHandler(async (req, res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    let keys = [];
    if (Array.isArray(req.body)) {
        keys = await service.updateBatch(req.body);
    }
    else if (req.body.keys) {
        keys = await service.updateMany(req.body.keys, req.body.data);
    }
    else {
        const sanitizedQuery = sanitizeQuery(req.body.query, req.accountability);
        keys = await service.updateByQuery(sanitizedQuery, req.body.data);
    }
    try {
        const result = await service.readMany(keys, req.sanitizedQuery);
        res.locals['payload'] = { data: result };
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
router.patch('/:pk', asyncHandler(async (req, res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    const primaryKey = await service.updateOne(req.params['pk'], req.body);
    try {
        const item = await service.readOne(primaryKey, req.sanitizedQuery);
        res.locals['payload'] = { data: item || null };
    }
    catch (error) {
        if (isDirectusError(error, ErrorCode.Forbidden)) {
            return next();
        }
        throw error;
    }
    return next();
}), respond);
router.delete('/', asyncHandler(async (req, _res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    if (Array.isArray(req.body)) {
        await service.deleteMany(req.body);
    }
    else if (req.body.keys) {
        await service.deleteMany(req.body.keys);
    }
    else {
        const sanitizedQuery = sanitizeQuery(req.body.query, req.accountability);
        await service.deleteByQuery(sanitizedQuery);
    }
    return next();
}), respond);
router.delete('/:pk', asyncHandler(async (req, _res, next) => {
    const service = new DashboardsService({
        accountability: req.accountability,
        schema: req.schema,
    });
    await service.deleteOne(req.params['pk']);
    return next();
}), respond);
export default router;
