"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sendInParallel = sendInParallel;
exports.sendMessage = sendMessage;
var _dgram = _interopRequireDefault(require("dgram"));
var _net = _interopRequireDefault(require("net"));
var _nodeUrl = _interopRequireDefault(require("node:url"));
var _abortError = _interopRequireDefault(require("./errors/abort-error"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
async function sendInParallel(addresses, port, request, signal) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  return await new Promise((resolve, reject) => {
    const sockets = [];
    let errorCount = 0;
    const onError = err => {
      errorCount++;
      if (errorCount === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        clearSockets();
        reject(err);
      }
    };
    const onMessage = message => {
      signal.removeEventListener('abort', onAbort);
      clearSockets();
      resolve(message);
    };
    const onAbort = () => {
      clearSockets();
      reject(new _abortError.default());
    };
    const clearSockets = () => {
      for (const socket of sockets) {
        socket.removeListener('error', onError);
        socket.removeListener('message', onMessage);
        socket.close();
      }
    };
    signal.addEventListener('abort', onAbort, {
      once: true
    });
    for (let j = 0; j < addresses.length; j++) {
      const udpType = addresses[j].family === 6 ? 'udp6' : 'udp4';
      const socket = _dgram.default.createSocket(udpType);
      sockets.push(socket);
      socket.on('error', onError);
      socket.on('message', onMessage);
      socket.send(request, 0, request.length, port, addresses[j].address);
    }
  });
}
async function sendMessage(host, port, lookup, signal, request) {
  if (signal.aborted) {
    throw new _abortError.default();
  }
  let addresses;
  if (_net.default.isIP(host)) {
    addresses = [{
      address: host,
      family: _net.default.isIPv6(host) ? 6 : 4
    }];
  } else {
    addresses = await new Promise((resolve, reject) => {
      const onAbort = () => {
        reject(new _abortError.default());
      };
      signal.addEventListener('abort', onAbort);
      lookup(_nodeUrl.default.domainToASCII(host), {
        all: true
      }, (err, addresses) => {
        signal.removeEventListener('abort', onAbort);
        err ? reject(err) : resolve(addresses);
      });
    });
  }
  return await sendInParallel(addresses, port, request, signal);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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