import { SchemaHelper } from '../types.js';
import { prepQueryParams } from '../utils/prep-query-params.js';
export class SchemaHelperOracle extends SchemaHelper {
    async changeToType(table, column, type, options = {}) {
        await this.changeToTypeByCopy(table, column, type, options);
    }
    castA2oPrimaryKey() {
        return 'CAST(?? AS VARCHAR2(255))';
    }
    preRelationChange(relation) {
        if (relation.collection === relation.related_collection) {
            // Constraints are not allowed on self referencing relationships
            // Setting NO ACTION throws - ORA-00905: missing keyword
            if (relation.schema?.on_delete) {
                relation.schema.on_delete = null;
            }
        }
    }
    processFieldType(field) {
        if (field.type === 'integer') {
            if (field.schema?.numeric_precision === 20) {
                return 'bigInteger';
            }
            else if (field.schema?.numeric_precision === 1) {
                return 'boolean';
            }
            else if (field.schema?.numeric_precision || field.schema?.numeric_scale) {
                return 'decimal';
            }
        }
        return field.type;
    }
    async getDatabaseSize() {
        try {
            const result = await this.knex.raw('select SUM(bytes) from dba_segments');
            return result[0]?.['SUM(BYTES)'] ? Number(result[0]?.['SUM(BYTES)']) : null;
        }
        catch {
            return null;
        }
    }
    prepQueryParams(queryParams) {
        return prepQueryParams(queryParams, { format: (index) => `:${index + 1}` });
    }
    prepBindings(bindings) {
        // Create an object with keys 1, 2, 3, ... and the bindings as values
        // This will use the "named" binding syntax in the oracledb driver instead of the positional binding
        return Object.fromEntries(bindings.map((binding, index) => [index + 1, binding]));
    }
    addInnerSortFieldsToGroupBy(groupByFields, sortRecords, _hasRelationalSort) {
        /*
        Oracle requires all selected columns that are not aggregated over to be present in the GROUP BY clause
        aliases can not be used before version 23c.

        > If you also specify a group_by_clause in this statement, then this select list can contain only the following
          types of expressions:
                * Constants
                * Aggregate functions and the functions USER, UID, and SYSDATE
                * Expressions identical to those in the group_by_clause. If the group_by_clause is in a subquery,
                  then all columns in the select list of the subquery must match the GROUP BY columns in the subquery.
                  If the select list and GROUP BY columns of a top-level query or of a subquery do not match,
                  then the statement results in ORA-00979.
                * Expressions involving the preceding expressions that evaluate to the same value for all rows in a group

        https://docs.oracle.com/en/database/oracle/oracle-database/19/sqlrf/SELECT.html
         */
        if (sortRecords.length > 0) {
            groupByFields.push(...sortRecords.map(({ column }) => column));
        }
    }
}
